import { Protocol } from '@uniswap/router-sdk';
import { Percent } from '@uniswap/sdk-core';
import { Pair } from '@uniswap/v2-sdk';
import { Pool as V3Pool } from '@uniswap/v3-sdk';
import { Pool as V4Pool } from '@uniswap/v4-sdk';
import _ from 'lodash';
import { V3_CORE_FACTORY_ADDRESSES } from './addresses';
import { CurrencyAmount, V4_ETH_WETH_FAKE_POOL } from '.';
export const routeToTokens = (route) => {
    switch (route.protocol) {
        case Protocol.V4:
            return route.currencyPath;
        case Protocol.V3:
            return route.tokenPath;
        case Protocol.V2:
        case Protocol.MIXED:
            return route.path;
        default:
            throw new Error(`Unsupported route ${JSON.stringify(route)}`);
    }
};
export const routeToPools = (route) => {
    switch (route.protocol) {
        case Protocol.V4:
        case Protocol.V3:
        case Protocol.MIXED:
            return route.pools;
        case Protocol.V2:
            return route.pairs;
        default:
            throw new Error(`Unsupported route ${JSON.stringify(route)}`);
    }
};
export const poolToString = (pool) => {
    if (pool instanceof V4Pool) {
        return ` -- ${pool.fee / 10000}% [${V4Pool.getPoolId(pool.token0, pool.token1, pool.fee, pool.tickSpacing, pool.hooks)}]`;
    }
    else if (pool instanceof V3Pool) {
        return ` -- ${pool.fee / 10000}% [${V3Pool.getAddress(pool.token0, pool.token1, pool.fee, undefined, V3_CORE_FACTORY_ADDRESSES[pool.chainId])}]`;
    }
    else if (pool instanceof Pair) {
        return ` -- [${Pair.getAddress(pool.token0, pool.token1)}]`;
    }
    else {
        throw new Error(`Unsupported pool ${JSON.stringify(pool)}`);
    }
};
export const routeToString = (route) => {
    const routeStr = [];
    const tokens = routeToTokens(route);
    const tokenPath = _.map(tokens, (token) => `${token.symbol}`);
    const pools = routeToPools(route);
    const poolFeePath = _.map(pools, (pool) => {
        if (pool instanceof Pair) {
            return ` -- [${Pair.getAddress(pool.token0, pool.token1)}]`;
        }
        else if (pool instanceof V3Pool) {
            return ` -- ${pool.fee / 10000}% [${V3Pool.getAddress(pool.token0, pool.token1, pool.fee, undefined, V3_CORE_FACTORY_ADDRESSES[pool.chainId])}]`;
        }
        else if (pool instanceof V4Pool) {
            // Special case in the case of ETH/WETH fake pool
            // where we do not want to return the fake pool in the route string as it is not a real pool
            if (pool.tickSpacing ===
                V4_ETH_WETH_FAKE_POOL[pool.chainId].tickSpacing) {
                return ' --  ';
            }
            return ` -- ${pool.fee / 10000}% [${V4Pool.getPoolId(pool.token0, pool.token1, pool.fee, pool.tickSpacing, pool.hooks)}]`;
        }
        else {
            throw new Error(`Unsupported pool ${JSON.stringify(pool)}`);
        }
        return `${poolToString(pool)} --> `;
    });
    for (let i = 0; i < tokenPath.length; i++) {
        routeStr.push(tokenPath[i]);
        if (i < poolFeePath.length) {
            routeStr.push(poolFeePath[i]);
        }
    }
    return routeStr.join('');
};
export const routeAmountsToString = (routeAmounts) => {
    const total = _.reduce(routeAmounts, (total, cur) => {
        return total.add(cur.amount);
    }, CurrencyAmount.fromRawAmount(routeAmounts[0].amount.currency, 0));
    const routeStrings = _.map(routeAmounts, ({ protocol, route, amount }) => {
        const portion = amount.divide(total);
        const percent = new Percent(portion.numerator, portion.denominator);
        /// @dev special case for MIXED routes we want to show user friendly V2+V3 instead
        return `[${protocol == Protocol.MIXED ? 'V2 + V3 + V4' : protocol}] ${percent.toFixed(2)}% = ${routeToString(route)}`;
    });
    return _.join(routeStrings, ', ');
};
export function shouldWipeoutCachedRoutes(cachedRoutes, routingConfig) {
    // We want to roll out the mixed route with UR v1_2 with percent control,
    // along with the cached routes so that we can test the performance of the mixed route with UR v1_2ss
    if ((routingConfig === null || routingConfig === void 0 ? void 0 : routingConfig.enableMixedRouteWithUR1_2) &&
        (
        // In case of optimisticCachedRoutes, we don't want to wipe out the cache
        // This is because the upstream client will indicate that it's a perf sensitive (likely online) request,
        // such that we should still use the cached routes.
        // In case of routing-api,
        // when intent=quote, optimisticCachedRoutes will be true, it means it's an online quote request, and we should use the cached routes.
        // when intent=caching, optimisticCachedRoutes will be false, it means it's an async routing lambda invocation for the benefit of
        // non-perf-sensitive, so that we can nullify the retrieved cached routes, if certain condition meets.
        routingConfig === null || routingConfig === void 0 ? void 0 : routingConfig.optimisticCachedRoutes)) {
        return false;
    }
    const containsExcludedProtocolPools = cachedRoutes === null || cachedRoutes === void 0 ? void 0 : cachedRoutes.routes.find((route) => {
        switch (route.protocol) {
            case Protocol.MIXED:
                return (route.route.pools.filter((pool) => {
                    return poolIsInExcludedProtocols(pool, routingConfig === null || routingConfig === void 0 ? void 0 : routingConfig.excludedProtocolsFromMixed);
                }).length > 0);
            default:
                return false;
        }
    });
    return containsExcludedProtocolPools !== undefined;
}
function poolIsInExcludedProtocols(pool, excludedProtocolsFromMixed) {
    var _a, _b, _c;
    if (pool instanceof V4Pool) {
        return (_a = excludedProtocolsFromMixed === null || excludedProtocolsFromMixed === void 0 ? void 0 : excludedProtocolsFromMixed.includes(Protocol.V4)) !== null && _a !== void 0 ? _a : false;
    }
    else if (pool instanceof V3Pool) {
        return (_b = excludedProtocolsFromMixed === null || excludedProtocolsFromMixed === void 0 ? void 0 : excludedProtocolsFromMixed.includes(Protocol.V3)) !== null && _b !== void 0 ? _b : false;
    }
    else if (pool instanceof Pair) {
        return (_c = excludedProtocolsFromMixed === null || excludedProtocolsFromMixed === void 0 ? void 0 : excludedProtocolsFromMixed.includes(Protocol.V2)) !== null && _c !== void 0 ? _c : false;
    }
    else {
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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